define([
    'underscore',
    'App',
    'backbone',
    'marionette',
    'modules/form/form-view-mixin',
    'modules/page/popup/popup-layout-view',
    'text!modules/preferences/preferences.html',
    'modules/preferences/email-preferences-section-view',
    'modules/notification-preferences/notification-preferences-module',
    'User',
],
function (
    _,
    app,
    Backbone,
    Marionette,
    Mixin,
    PopupView,
    template,
    EmailPreferencesSectionView,
    notificationPreferencesModule,
    User
) {
    'use strict';

    return PopupView.extend(Mixin.form({
        'template': _.template(template),
        'id': 'preferences',
        'events': {
            'popupbeforeposition': 'styleMaxHeight',
            'popupafteropen': 'focusOnHeader',
            'popupafterclose': 'destroy',
            'submit #email-preferences-form': 'preventDefault',
            'mousedown #preferences-save-btn': 'preventDefault', /* to prevent blur from interfering with saving */
            'click #preferences-save-btn': 'savePreferences',
            'click #preferences-cancel-btn': 'closePopup',
        },
        'regions': {'emailPreferencesRegion': '.email-preferences-section'},
        '_idPrefix': 'modal-',

        'initialize': function(options) {
            if (_.isUndefined(this.model)) {
                // store data in clone of notification preferences until successful save
                this.model = notificationPreferencesModule.notificationPreferences.clone();
            }

            this._initializeElAfterClose(options);
        },

        'onRender': function () {
            // from popup-view
            this.$el.popup({dismissible: false});
            this.addAriaRole();

            this.initializeFormValidation({errorLabelContainer: '.error-list'});

            this.showChildView('emailPreferencesRegion', new EmailPreferencesSectionView({
                model: this.model,
                idPrefix: this._idPrefix,
            }));

            this.hideErrorSection();

        },

        'preventDefault': function(e) {
            e.preventDefault();
        },

        'resetFormValidation': function() {
            this.hideErrorSection();
            this.validator.resetForm();
        },

        'savePreferences': function() {
            /* preventDefault on mousedown prevents subsequent events from firing,
                 * including blur, which would update the model */
            this.updateEmailPrefsModel();
            this.saveEmailPreferences();
        },

        'updateEmailPrefsModel': function() {
            var notificationFrequency = this.model.get('emailAllowed') ? 'Each new message' : 'Never';
            this.model.set('emailAddress', this.$el.find('#' + this._idPrefix + 'emailAddress').val());
            this.model.set('notificationFrequency', notificationFrequency);
        },

        'saveEmailPreferences': function(e) {
            this.resetFormValidation();
            if (this.$el.find('form').valid()) {

                // save to NotificationPreferences singleton
                notificationPreferencesModule.notificationPreferences.save(
                    this.model.attributes,
                    {
                        url: app.resources.get('notification-preference').formatHref({
                            'assigning-authority': User.get('idType'),
                            'patient-id': User.get('id'),
                        }),
                        type: 'PUT',
                    }
                );

                this.closePopup();

            } else {
                this.showErrorSection();
            }
        },

        'showErrorSection': function() {
            if (!this.isDestroyed()) {
                this.$el.find('.error-section').removeClass('hidden');
                this.$el.find('#error-list-heading').focus();
            }
        },
        'hideErrorSection': function() {
            if (!this.isDestroyed()) {
                this.$el.find('.error-section').addClass('hidden');
            }
        },
    }));
});
